<?php
/**
 * PDPrint Admin - Dashboard
 */

require_once '../includes/config.php';

// Verificar se está logado
if (!is_admin_logged()) {
    redirect('index.php');
}

$db = new Database();

// Estatísticas gerais
$stats = [
    'pedidos_hoje' => $db->raw("SELECT COUNT(*) as total FROM pedidos WHERE DATE(created_at) = CURDATE()")[0]['total'] ?? 0,
    'vendas_mes' => $db->raw("SELECT SUM(total) as total FROM pedidos WHERE MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE()) AND status != 'cancelado'")[0]['total'] ?? 0,
    'produtos_cadastrados' => $db->raw("SELECT COUNT(*) as total FROM produtos WHERE ativo = 1")[0]['total'] ?? 0,
    'clientes_cadastrados' => $db->raw("SELECT COUNT(*) as total FROM clientes WHERE ativo = 1")[0]['total'] ?? 0
];

// Pedidos recentes
$pedidos_recentes = $db->raw("
    SELECT p.*, c.nome as cliente_nome, f.nome as franquia_nome 
    FROM pedidos p 
    LEFT JOIN clientes c ON p.cliente_id = c.id 
    LEFT JOIN franquias f ON p.franquia_id = f.id 
    ORDER BY p.created_at DESC 
    LIMIT 10
");

// Produtos com estoque baixo
$estoque_baixo = $db->raw("
    SELECT p.nome, p.estoque_minimo, SUM(e.quantidade) as estoque_atual
    FROM produtos p
    LEFT JOIN estoque e ON p.id = e.produto_id
    WHERE p.controlar_estoque = 1 AND p.ativo = 1
    GROUP BY p.id
    HAVING estoque_atual <= p.estoque_minimo OR estoque_atual IS NULL
    ORDER BY estoque_atual ASC
    LIMIT 10
");

$user = get_admin_user();
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - PDPrint Admin</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #f8fafc;
            color: #333;
        }
        
        .header {
            background: white;
            padding: 20px 30px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .logo h1 {
            color: #667eea;
            font-size: 1.8rem;
            font-weight: 700;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .user-info span {
            color: #666;
        }
        
        .btn {
            background: #667eea;
            color: white;
            padding: 8px 16px;
            border: none;
            border-radius: 6px;
            text-decoration: none;
            font-size: 14px;
            cursor: pointer;
            transition: background 0.3s;
        }
        
        .btn:hover {
            background: #5a6fd8;
        }
        
        .btn-danger {
            background: #e53e3e;
        }
        
        .btn-danger:hover {
            background: #c53030;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 30px;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 40px;
        }
        
        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            text-align: center;
        }
        
        .stat-number {
            font-size: 2.5rem;
            font-weight: 700;
            color: #667eea;
            margin-bottom: 10px;
        }
        
        .stat-label {
            color: #666;
            font-size: 1rem;
        }
        
        .content-grid {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 30px;
        }
        
        .card {
            background: white;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .card-header {
            padding: 20px 25px;
            border-bottom: 1px solid #eee;
            font-weight: 600;
            color: #333;
        }
        
        .card-body {
            padding: 25px;
        }
        
        .table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .table th,
        .table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        
        .table th {
            background: #f8fafc;
            font-weight: 600;
            color: #333;
        }
        
        .status {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: 500;
        }
        
        .status-pendente {
            background: #fef5e7;
            color: #d69e2e;
        }
        
        .status-confirmado {
            background: #e6fffa;
            color: #319795;
        }
        
        .status-cancelado {
            background: #fed7d7;
            color: #e53e3e;
        }
        
        .nav-menu {
            background: white;
            padding: 0 30px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.05);
        }
        
        .nav-menu ul {
            list-style: none;
            display: flex;
            gap: 30px;
        }
        
        .nav-menu a {
            display: block;
            padding: 15px 0;
            color: #666;
            text-decoration: none;
            font-weight: 500;
            border-bottom: 3px solid transparent;
            transition: all 0.3s;
        }
        
        .nav-menu a:hover,
        .nav-menu a.active {
            color: #667eea;
            border-bottom-color: #667eea;
        }
        
        .alert {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .alert-warning {
            background: #fef5e7;
            color: #d69e2e;
            border: 1px solid #f6e05e;
        }
        
        @media (max-width: 768px) {
            .content-grid {
                grid-template-columns: 1fr;
            }
            
            .nav-menu ul {
                flex-direction: column;
                gap: 0;
            }
            
            .container {
                padding: 20px;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="logo">
            <h1>PDPrint Admin</h1>
        </div>
        <div class="user-info">
            <span>Olá, <?= htmlspecialchars($user['nome']) ?></span>
            <a href="logout.php" class="btn btn-danger">Sair</a>
        </div>
    </div>
    
    <nav class="nav-menu">
        <ul>
            <li><a href="dashboard.php" class="active">Dashboard</a></li>
            <li><a href="produtos.php">Produtos</a></li>
            <li><a href="pedidos.php">Pedidos</a></li>
            <li><a href="clientes.php">Clientes</a></li>
            <li><a href="estoque.php">Estoque</a></li>
            <li><a href="financeiro.php">Financeiro</a></li>
            <li><a href="relatorios.php">Relatórios</a></li>
            <?php if ($user['tipo'] === 'super_admin'): ?>
            <li><a href="franquias.php">Franquias</a></li>
            <li><a href="usuarios.php">Usuários</a></li>
            <?php endif; ?>
        </ul>
    </nav>
    
    <div class="container">
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-number"><?= $stats['pedidos_hoje'] ?></div>
                <div class="stat-label">Pedidos Hoje</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= format_currency($stats['vendas_mes']) ?></div>
                <div class="stat-label">Vendas do Mês</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= $stats['produtos_cadastrados'] ?></div>
                <div class="stat-label">Produtos Ativos</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= $stats['clientes_cadastrados'] ?></div>
                <div class="stat-label">Clientes Cadastrados</div>
            </div>
        </div>
        
        <div class="content-grid">
            <div class="card">
                <div class="card-header">
                    Pedidos Recentes
                </div>
                <div class="card-body">
                    <?php if (empty($pedidos_recentes)): ?>
                        <p>Nenhum pedido encontrado.</p>
                    <?php else: ?>
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Pedido</th>
                                    <th>Cliente</th>
                                    <th>Total</th>
                                    <th>Status</th>
                                    <th>Data</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($pedidos_recentes as $pedido): ?>
                                <tr>
                                    <td>#<?= htmlspecialchars($pedido['numero']) ?></td>
                                    <td><?= htmlspecialchars($pedido['cliente_nome']) ?></td>
                                    <td><?= format_currency($pedido['total']) ?></td>
                                    <td><span class="status status-<?= $pedido['status'] ?>"><?= ucfirst($pedido['status']) ?></span></td>
                                    <td><?= format_date($pedido['created_at']) ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    Estoque Baixo
                </div>
                <div class="card-body">
                    <?php if (empty($estoque_baixo)): ?>
                        <p>Todos os produtos estão com estoque adequado.</p>
                    <?php else: ?>
                        <div class="alert alert-warning">
                            <strong>Atenção!</strong> Produtos com estoque baixo ou zerado.
                        </div>
                        <?php foreach ($estoque_baixo as $produto): ?>
                        <div style="padding: 10px 0; border-bottom: 1px solid #eee;">
                            <strong><?= htmlspecialchars($produto['nome']) ?></strong><br>
                            <small>Estoque: <?= $produto['estoque_atual'] ?? 0 ?> | Mínimo: <?= $produto['estoque_minimo'] ?></small>
                        </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</body>
</html>

