<?php
/**
 * PDPrint Admin - Gestão de Produtos
 */

require_once '../includes/config.php';

if (!is_admin_logged()) {
    redirect('index.php');
}

$db = new Database();
$user = get_admin_user();

// Processar ações
if (is_post()) {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'delete' && isset($_POST['id'])) {
        $id = (int)$_POST['id'];
        $produto = $db->select('produtos', ['id' => $id])[0] ?? null;
        
        if ($produto) {
            $db->update('produtos', ['ativo' => 0], ['id' => $id]);
            log_activity('Produto desativado', 'produtos', $id, $produto);
            $success = 'Produto removido com sucesso!';
        }
    }
}

// Filtros
$search = $_GET['search'] ?? '';
$categoria = $_GET['categoria'] ?? '';
$page = (int)($_GET['page'] ?? 1);

// Buscar categorias para filtro
$categorias = $db->select('categorias', ['ativo' => 1], '*', 'nome ASC');

// Montar query de busca
$where = ['ativo' => 1];
if ($search) {
    $produtos = $db->raw("
        SELECT p.*, c.nome as categoria_nome 
        FROM produtos p 
        LEFT JOIN categorias c ON p.categoria_id = c.id 
        WHERE p.ativo = 1 AND (p.nome LIKE :search OR p.sku LIKE :search)
        ORDER BY p.created_at DESC
        LIMIT " . (($page - 1) * ITEMS_PER_PAGE) . ", " . ITEMS_PER_PAGE,
        [':search' => "%{$search}%"]
    );
    
    $total = $db->raw("
        SELECT COUNT(*) as total FROM produtos 
        WHERE ativo = 1 AND (nome LIKE :search OR sku LIKE :search)",
        [':search' => "%{$search}%"]
    )[0]['total'];
} else {
    if ($categoria) {
        $where['categoria_id'] = $categoria;
    }
    
    $result = $db->paginate('produtos p LEFT JOIN categorias c ON p.categoria_id = c.id', 
                           $where, 'p.*, c.nome as categoria_nome', 'p.created_at DESC', $page);
    $produtos = $result['data'];
    $total = $result['total'];
}

$total_pages = ceil($total / ITEMS_PER_PAGE);
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Produtos - PDPrint Admin</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #f8fafc;
            color: #333;
        }
        
        .header {
            background: white;
            padding: 20px 30px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .logo h1 {
            color: #667eea;
            font-size: 1.8rem;
            font-weight: 700;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .btn {
            background: #667eea;
            color: white;
            padding: 8px 16px;
            border: none;
            border-radius: 6px;
            text-decoration: none;
            font-size: 14px;
            cursor: pointer;
            transition: background 0.3s;
            display: inline-block;
        }
        
        .btn:hover {
            background: #5a6fd8;
        }
        
        .btn-success {
            background: #38a169;
        }
        
        .btn-success:hover {
            background: #2f855a;
        }
        
        .btn-danger {
            background: #e53e3e;
        }
        
        .btn-danger:hover {
            background: #c53030;
        }
        
        .btn-sm {
            padding: 4px 8px;
            font-size: 12px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 30px;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
        }
        
        .page-title {
            font-size: 2rem;
            font-weight: 700;
            color: #333;
        }
        
        .filters {
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        
        .filters-row {
            display: flex;
            gap: 15px;
            align-items: end;
        }
        
        .form-group {
            flex: 1;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
            color: #333;
        }
        
        .form-control {
            width: 100%;
            padding: 10px 12px;
            border: 2px solid #e1e5e9;
            border-radius: 6px;
            font-size: 14px;
        }
        
        .form-control:focus {
            outline: none;
            border-color: #667eea;
        }
        
        .card {
            background: white;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .table th,
        .table td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        
        .table th {
            background: #f8fafc;
            font-weight: 600;
            color: #333;
        }
        
        .table tr:hover {
            background: #f8fafc;
        }
        
        .product-image {
            width: 50px;
            height: 50px;
            border-radius: 6px;
            object-fit: cover;
            background: #f0f0f0;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #999;
        }
        
        .status {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: 500;
        }
        
        .status-ativo {
            background: #e6fffa;
            color: #319795;
        }
        
        .status-inativo {
            background: #fed7d7;
            color: #e53e3e;
        }
        
        .actions {
            display: flex;
            gap: 5px;
        }
        
        .pagination {
            display: flex;
            justify-content: center;
            gap: 10px;
            margin-top: 20px;
        }
        
        .pagination a,
        .pagination span {
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            text-decoration: none;
            color: #333;
        }
        
        .pagination a:hover {
            background: #f0f0f0;
        }
        
        .pagination .current {
            background: #667eea;
            color: white;
            border-color: #667eea;
        }
        
        .alert {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .alert-success {
            background: #e6fffa;
            color: #319795;
            border: 1px solid #81e6d9;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #666;
        }
        
        .empty-state h3 {
            font-size: 1.5rem;
            margin-bottom: 10px;
        }
        
        @media (max-width: 768px) {
            .filters-row {
                flex-direction: column;
            }
            
            .page-header {
                flex-direction: column;
                gap: 20px;
                align-items: stretch;
            }
            
            .table {
                font-size: 14px;
            }
            
            .table th,
            .table td {
                padding: 10px 8px;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="logo">
            <h1>PDPrint Admin</h1>
        </div>
        <div class="user-info">
            <span>Olá, <?= htmlspecialchars($user['nome']) ?></span>
            <a href="dashboard.php" class="btn">Dashboard</a>
            <a href="logout.php" class="btn btn-danger">Sair</a>
        </div>
    </div>
    
    <div class="container">
        <div class="page-header">
            <h1 class="page-title">Produtos</h1>
            <a href="produto-form.php" class="btn btn-success">+ Novo Produto</a>
        </div>
        
        <?php if (isset($success)): ?>
            <div class="alert alert-success">
                <?= htmlspecialchars($success) ?>
            </div>
        <?php endif; ?>
        
        <div class="filters">
            <form method="GET" class="filters-row">
                <div class="form-group">
                    <label for="search">Buscar</label>
                    <input type="text" id="search" name="search" class="form-control" 
                           placeholder="Nome ou SKU do produto" value="<?= htmlspecialchars($search) ?>">
                </div>
                
                <div class="form-group">
                    <label for="categoria">Categoria</label>
                    <select id="categoria" name="categoria" class="form-control">
                        <option value="">Todas as categorias</option>
                        <?php foreach ($categorias as $cat): ?>
                        <option value="<?= $cat['id'] ?>" <?= $categoria == $cat['id'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($cat['nome']) ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <button type="submit" class="btn">Filtrar</button>
                    <a href="produtos.php" class="btn" style="background: #666;">Limpar</a>
                </div>
            </form>
        </div>
        
        <div class="card">
            <?php if (empty($produtos)): ?>
                <div class="empty-state">
                    <h3>Nenhum produto encontrado</h3>
                    <p>Comece cadastrando seu primeiro produto.</p>
                    <a href="produto-form.php" class="btn btn-success">+ Cadastrar Produto</a>
                </div>
            <?php else: ?>
                <table class="table">
                    <thead>
                        <tr>
                            <th>Imagem</th>
                            <th>Nome</th>
                            <th>SKU</th>
                            <th>Categoria</th>
                            <th>Preço</th>
                            <th>Status</th>
                            <th>Ações</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($produtos as $produto): ?>
                        <tr>
                            <td>
                                <div class="product-image">
                                    <?php if ($produto['imagem_principal']): ?>
                                        <img src="<?= UPLOAD_URL . $produto['imagem_principal'] ?>" 
                                             alt="<?= htmlspecialchars($produto['nome']) ?>" 
                                             style="width: 50px; height: 50px; object-fit: cover; border-radius: 6px;">
                                    <?php else: ?>
                                        📦
                                    <?php endif; ?>
                                </div>
                            </td>
                            <td>
                                <strong><?= htmlspecialchars($produto['nome']) ?></strong>
                                <?php if ($produto['descricao_curta']): ?>
                                <br><small style="color: #666;"><?= htmlspecialchars(substr($produto['descricao_curta'], 0, 50)) ?>...</small>
                                <?php endif; ?>
                            </td>
                            <td><?= htmlspecialchars($produto['sku'] ?: '-') ?></td>
                            <td><?= htmlspecialchars($produto['categoria_nome']) ?></td>
                            <td>
                                <?php if ($produto['preco_promocional']): ?>
                                    <span style="text-decoration: line-through; color: #999; font-size: 0.9rem;">
                                        <?= format_currency($produto['preco']) ?>
                                    </span><br>
                                    <strong><?= format_currency($produto['preco_promocional']) ?></strong>
                                <?php else: ?>
                                    <?= format_currency($produto['preco']) ?>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="status status-<?= $produto['ativo'] ? 'ativo' : 'inativo' ?>">
                                    <?= $produto['ativo'] ? 'Ativo' : 'Inativo' ?>
                                </span>
                            </td>
                            <td>
                                <div class="actions">
                                    <a href="produto-form.php?id=<?= $produto['id'] ?>" class="btn btn-sm">Editar</a>
                                    <form method="POST" style="display: inline;" 
                                          onsubmit="return confirm('Tem certeza que deseja remover este produto?')">
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="id" value="<?= $produto['id'] ?>">
                                        <button type="submit" class="btn btn-sm btn-danger">Remover</button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <?php if ($total_pages > 1): ?>
                <div class="pagination">
                    <?php if ($page > 1): ?>
                        <a href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&categoria=<?= $categoria ?>">« Anterior</a>
                    <?php endif; ?>
                    
                    <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                        <?php if ($i == $page): ?>
                            <span class="current"><?= $i ?></span>
                        <?php else: ?>
                            <a href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&categoria=<?= $categoria ?>"><?= $i ?></a>
                        <?php endif; ?>
                    <?php endfor; ?>
                    
                    <?php if ($page < $total_pages): ?>
                        <a href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&categoria=<?= $categoria ?>">Próxima »</a>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>

