<?php
/**
 * PDPrint - Classe Database
 * Versão: 1.0
 */

class Database {
    private $pdo;
    private $stmt;
    
    public function __construct() {
        $this->connect();
    }
    
    /**
     * Conectar ao banco de dados
     */
    private function connect() {
        try {
            $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=' . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ];
            
            $this->pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            die('Erro de conexão: ' . $e->getMessage());
        }
    }
    
    /**
     * Preparar query
     */
    public function query($sql) {
        $this->stmt = $this->pdo->prepare($sql);
        return $this;
    }
    
    /**
     * Bind de parâmetros
     */
    public function bind($param, $value, $type = null) {
        if (is_null($type)) {
            switch (true) {
                case is_int($value):
                    $type = PDO::PARAM_INT;
                    break;
                case is_bool($value):
                    $type = PDO::PARAM_BOOL;
                    break;
                case is_null($value):
                    $type = PDO::PARAM_NULL;
                    break;
                default:
                    $type = PDO::PARAM_STR;
            }
        }
        
        $this->stmt->bindValue($param, $value, $type);
        return $this;
    }
    
    /**
     * Executar query
     */
    public function execute() {
        return $this->stmt->execute();
    }
    
    /**
     * Buscar múltiplos registros
     */
    public function fetchAll() {
        $this->execute();
        return $this->stmt->fetchAll();
    }
    
    /**
     * Buscar um registro
     */
    public function fetch() {
        $this->execute();
        return $this->stmt->fetch();
    }
    
    /**
     * Contar registros
     */
    public function rowCount() {
        return $this->stmt->rowCount();
    }
    
    /**
     * Último ID inserido
     */
    public function lastInsertId() {
        return $this->pdo->lastInsertId();
    }
    
    /**
     * Iniciar transação
     */
    public function beginTransaction() {
        return $this->pdo->beginTransaction();
    }
    
    /**
     * Commit da transação
     */
    public function commit() {
        return $this->pdo->commit();
    }
    
    /**
     * Rollback da transação
     */
    public function rollback() {
        return $this->pdo->rollback();
    }
    
    /**
     * SELECT simplificado
     */
    public function select($table, $where = [], $fields = '*', $order = null, $limit = null) {
        $sql = "SELECT {$fields} FROM {$table}";
        
        if (!empty($where)) {
            $conditions = [];
            foreach ($where as $key => $value) {
                $conditions[] = "{$key} = :{$key}";
            }
            $sql .= " WHERE " . implode(' AND ', $conditions);
        }
        
        if ($order) {
            $sql .= " ORDER BY {$order}";
        }
        
        if ($limit) {
            $sql .= " LIMIT {$limit}";
        }
        
        $this->query($sql);
        
        foreach ($where as $key => $value) {
            $this->bind(":{$key}", $value);
        }
        
        return $this->fetchAll();
    }
    
    /**
     * INSERT simplificado
     */
    public function insert($table, $data) {
        $fields = implode(', ', array_keys($data));
        $placeholders = ':' . implode(', :', array_keys($data));
        
        $sql = "INSERT INTO {$table} ({$fields}) VALUES ({$placeholders})";
        
        $this->query($sql);
        
        foreach ($data as $key => $value) {
            $this->bind(":{$key}", $value);
        }
        
        $this->execute();
        return $this->lastInsertId();
    }
    
    /**
     * UPDATE simplificado
     */
    public function update($table, $data, $where) {
        $set = [];
        foreach ($data as $key => $value) {
            $set[] = "{$key} = :set_{$key}";
        }
        
        $conditions = [];
        foreach ($where as $key => $value) {
            $conditions[] = "{$key} = :where_{$key}";
        }
        
        $sql = "UPDATE {$table} SET " . implode(', ', $set) . 
               " WHERE " . implode(' AND ', $conditions);
        
        $this->query($sql);
        
        foreach ($data as $key => $value) {
            $this->bind(":set_{$key}", $value);
        }
        
        foreach ($where as $key => $value) {
            $this->bind(":where_{$key}", $value);
        }
        
        $this->execute();
        return $this->rowCount();
    }
    
    /**
     * DELETE simplificado
     */
    public function delete($table, $where) {
        $conditions = [];
        foreach ($where as $key => $value) {
            $conditions[] = "{$key} = :{$key}";
        }
        
        $sql = "DELETE FROM {$table} WHERE " . implode(' AND ', $conditions);
        
        $this->query($sql);
        
        foreach ($where as $key => $value) {
            $this->bind(":{$key}", $value);
        }
        
        $this->execute();
        return $this->rowCount();
    }
    
    /**
     * Buscar com paginação
     */
    public function paginate($table, $where = [], $fields = '*', $order = null, $page = 1, $per_page = ITEMS_PER_PAGE) {
        // Contar total de registros
        $count_sql = "SELECT COUNT(*) as total FROM {$table}";
        
        if (!empty($where)) {
            $conditions = [];
            foreach ($where as $key => $value) {
                $conditions[] = "{$key} = :{$key}";
            }
            $count_sql .= " WHERE " . implode(' AND ', $conditions);
        }
        
        $this->query($count_sql);
        
        foreach ($where as $key => $value) {
            $this->bind(":{$key}", $value);
        }
        
        $total = $this->fetch()['total'];
        
        // Buscar registros da página
        $offset = ($page - 1) * $per_page;
        
        $sql = "SELECT {$fields} FROM {$table}";
        
        if (!empty($where)) {
            $conditions = [];
            foreach ($where as $key => $value) {
                $conditions[] = "{$key} = :{$key}";
            }
            $sql .= " WHERE " . implode(' AND ', $conditions);
        }
        
        if ($order) {
            $sql .= " ORDER BY {$order}";
        }
        
        $sql .= " LIMIT {$per_page} OFFSET {$offset}";
        
        $this->query($sql);
        
        foreach ($where as $key => $value) {
            $this->bind(":{$key}", $value);
        }
        
        $data = $this->fetchAll();
        
        return [
            'data' => $data,
            'total' => $total,
            'page' => $page,
            'per_page' => $per_page,
            'total_pages' => ceil($total / $per_page),
            'has_next' => $page < ceil($total / $per_page),
            'has_prev' => $page > 1
        ];
    }
    
    /**
     * Executar SQL customizado
     */
    public function raw($sql, $params = []) {
        $this->query($sql);
        
        foreach ($params as $key => $value) {
            $this->bind($key, $value);
        }
        
        return $this->fetchAll();
    }
}
?>

