<?php
/**
 * PDPrint - Funções Auxiliares
 * Versão: 1.0
 */

/**
 * Sanitizar dados de entrada
 */
function sanitize($data) {
    if (is_array($data)) {
        return array_map('sanitize', $data);
    }
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

/**
 * Validar email
 */
function validate_email($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Gerar hash de senha
 */
function hash_password($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * Verificar senha
 */
function verify_password($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Gerar token aleatório
 */
function generate_token($length = 32) {
    return bin2hex(random_bytes($length));
}

/**
 * Formatar moeda
 */
function format_currency($value) {
    return CURRENCY_SYMBOL . ' ' . number_format($value, 2, ',', '.');
}

/**
 * Formatar data
 */
function format_date($date, $format = 'd/m/Y') {
    if (empty($date) || $date === '0000-00-00') {
        return '-';
    }
    return date($format, strtotime($date));
}

/**
 * Formatar data e hora
 */
function format_datetime($datetime, $format = 'd/m/Y H:i') {
    if (empty($datetime) || $datetime === '0000-00-00 00:00:00') {
        return '-';
    }
    return date($format, strtotime($datetime));
}

/**
 * Gerar slug
 */
function generate_slug($text) {
    $text = strtolower($text);
    $text = preg_replace('/[^a-z0-9\s-]/', '', $text);
    $text = preg_replace('/[\s-]+/', '-', $text);
    return trim($text, '-');
}

/**
 * Upload de arquivo
 */
function upload_file($file, $folder = 'geral') {
    if (!isset($file['tmp_name']) || empty($file['tmp_name'])) {
        return ['success' => false, 'message' => 'Nenhum arquivo enviado'];
    }
    
    $upload_dir = UPLOAD_PATH . $folder . '/';
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'pdf', 'doc', 'docx'];
    
    if (!in_array($file_extension, $allowed_extensions)) {
        return ['success' => false, 'message' => 'Tipo de arquivo não permitido'];
    }
    
    if ($file['size'] > MAX_FILE_SIZE) {
        return ['success' => false, 'message' => 'Arquivo muito grande'];
    }
    
    $filename = uniqid() . '.' . $file_extension;
    $filepath = $upload_dir . $filename;
    
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        return [
            'success' => true,
            'filename' => $filename,
            'path' => $folder . '/' . $filename,
            'url' => UPLOAD_URL . $folder . '/' . $filename
        ];
    }
    
    return ['success' => false, 'message' => 'Erro ao fazer upload'];
}

/**
 * Redimensionar imagem
 */
function resize_image($source, $destination, $width, $height) {
    $info = getimagesize($source);
    $mime = $info['mime'];
    
    switch ($mime) {
        case 'image/jpeg':
            $image = imagecreatefromjpeg($source);
            break;
        case 'image/png':
            $image = imagecreatefrompng($source);
            break;
        case 'image/gif':
            $image = imagecreatefromgif($source);
            break;
        default:
            return false;
    }
    
    $original_width = imagesx($image);
    $original_height = imagesy($image);
    
    $ratio = min($width / $original_width, $height / $original_height);
    $new_width = $original_width * $ratio;
    $new_height = $original_height * $ratio;
    
    $new_image = imagecreatetruecolor($new_width, $new_height);
    
    if ($mime === 'image/png' || $mime === 'image/gif') {
        imagealphablending($new_image, false);
        imagesavealpha($new_image, true);
    }
    
    imagecopyresampled($new_image, $image, 0, 0, 0, 0, 
                      $new_width, $new_height, $original_width, $original_height);
    
    switch ($mime) {
        case 'image/jpeg':
            imagejpeg($new_image, $destination, 90);
            break;
        case 'image/png':
            imagepng($new_image, $destination);
            break;
        case 'image/gif':
            imagegif($new_image, $destination);
            break;
    }
    
    imagedestroy($image);
    imagedestroy($new_image);
    
    return true;
}

/**
 * Enviar email
 */
function send_email($to, $subject, $message, $from = null) {
    $from = $from ?: SMTP_FROM;
    
    $headers = [
        'MIME-Version: 1.0',
        'Content-type: text/html; charset=UTF-8',
        'From: ' . SMTP_FROM_NAME . ' <' . $from . '>',
        'Reply-To: ' . $from,
        'X-Mailer: PHP/' . phpversion()
    ];
    
    return mail($to, $subject, $message, implode("\r\n", $headers));
}

/**
 * Gerar número de pedido
 */
function generate_order_number() {
    return date('Y') . str_pad(mt_rand(1, 999999), 6, '0', STR_PAD_LEFT);
}

/**
 * Calcular frete (simulação)
 */
function calculate_shipping($cep, $weight = 1) {
    // Simulação simples de cálculo de frete
    $base_price = 15.00;
    $weight_price = $weight * 2.50;
    
    return $base_price + $weight_price;
}

/**
 * Validar CPF
 */
function validate_cpf($cpf) {
    $cpf = preg_replace('/[^0-9]/', '', $cpf);
    
    if (strlen($cpf) != 11 || preg_match('/(\d)\1{10}/', $cpf)) {
        return false;
    }
    
    for ($t = 9; $t < 11; $t++) {
        for ($d = 0, $c = 0; $c < $t; $c++) {
            $d += $cpf[$c] * (($t + 1) - $c);
        }
        $d = ((10 * $d) % 11) % 10;
        if ($cpf[$c] != $d) {
            return false;
        }
    }
    
    return true;
}

/**
 * Validar CNPJ
 */
function validate_cnpj($cnpj) {
    $cnpj = preg_replace('/[^0-9]/', '', $cnpj);
    
    if (strlen($cnpj) != 14) {
        return false;
    }
    
    for ($i = 0, $j = 5, $soma = 0; $i < 12; $i++) {
        $soma += $cnpj[$i] * $j;
        $j = ($j == 2) ? 9 : $j - 1;
    }
    
    $resto = $soma % 11;
    
    if ($cnpj[12] != ($resto < 2 ? 0 : 11 - $resto)) {
        return false;
    }
    
    for ($i = 0, $j = 6, $soma = 0; $i < 13; $i++) {
        $soma += $cnpj[$i] * $j;
        $j = ($j == 2) ? 9 : $j - 1;
    }
    
    $resto = $soma % 11;
    
    return $cnpj[13] == ($resto < 2 ? 0 : 11 - $resto);
}

/**
 * Resposta JSON
 */
function json_response($data, $status_code = 200) {
    http_response_code($status_code);
    header('Content-Type: application/json; charset=UTF-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Redirect
 */
function redirect($url) {
    header('Location: ' . $url);
    exit;
}

/**
 * Verificar se usuário está logado (admin)
 */
function is_admin_logged() {
    return isset($_SESSION['admin_id']) && !empty($_SESSION['admin_id']);
}

/**
 * Verificar se cliente está logado (loja)
 */
function is_customer_logged() {
    return isset($_SESSION['customer_id']) && !empty($_SESSION['customer_id']);
}

/**
 * Obter dados do usuário logado (admin)
 */
function get_admin_user() {
    if (!is_admin_logged()) {
        return null;
    }
    
    $db = new Database();
    return $db->select('usuarios', ['id' => $_SESSION['admin_id']])[0] ?? null;
}

/**
 * Obter dados do cliente logado (loja)
 */
function get_customer() {
    if (!is_customer_logged()) {
        return null;
    }
    
    $db = new Database();
    return $db->select('clientes', ['id' => $_SESSION['customer_id']])[0] ?? null;
}

/**
 * Log de atividades
 */
function log_activity($acao, $tabela = null, $registro_id = null, $dados_anteriores = null, $dados_novos = null) {
    $db = new Database();
    
    $log_data = [
        'usuario_id' => $_SESSION['admin_id'] ?? null,
        'acao' => $acao,
        'tabela' => $tabela,
        'registro_id' => $registro_id,
        'dados_anteriores' => $dados_anteriores ? json_encode($dados_anteriores) : null,
        'dados_novos' => $dados_novos ? json_encode($dados_novos) : null,
        'ip' => get_client_ip(),
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null
    ];
    
    $db->insert('logs', $log_data);
}
?>

